//-----------------------------------------------------------------------------
//  Galv's Visibility Range
//-----------------------------------------------------------------------------
//  For: RPGMAKER MV
//  GALV_VisibilityRange.js
//-----------------------------------------------------------------------------
//  2015-12-19 - Version 1.0 - release
//-----------------------------------------------------------------------------
// Terms can be found at:
// galvs-scripts.com
//-----------------------------------------------------------------------------

var Imported = Imported || {};
Imported.Galv_VisibilityRange = true;

var Galv = Galv || {};        // Galv's main object
Galv.pCmd = Galv.pCmd || {};  // Plugin Command manager
Galv.VR = Galv.VR || {};      // Galv's plugin stuff

//-----------------------------------------------------------------------------
/*:
 * @plugindesc 使用一張圖片以及不透明度達到修改能見度效果
 *
 * @author Galv - galvs-scripts.com 繁化:RouZong
 *
 * @param Zoom Variable
 * @desc 設置光源焦距使用的變數編號
 * @default 0
 *
 * @param Opacity Variable
 * @desc 設置不透明度使用的變數編號
 * @default 0
 *
 * @param Z Position
 * @desc 該圖層的高低，太低就沒效果了
 * @default 10
 *
 * @help Galv's 能見度範圍
 * ----------------------------------------------------------------------------
 * 這個插件可以讓你顯示圖片在玩家上。
 * 這想法是源自於可見度範圍. 這圖片跟隨玩家，並可以放大
 * 並通過在插件的配置設定變數，然後使用"控制變數"事件來改變他們的不透明度。
 *注意：變數默認設定是0。為了使用這個功能你需要改變它們
 *
 * 控制 焦距 放大/縮小:
 * 設置 焦距 變數id.
 * 焦距等級公式:50% + variable%.
 * 舉例，我們將變數設置在 #1.
 * 當變數 #1 是小於0時, 能見度範圍將在其正常大小的50％被顯示.
 * 這是最小尺寸顯示.
 * 當變數設置為 50, 他將顯示 100% (50 + 50 = 100).
 * 
 * 不透明度變數範圍 0 - 255，控制能見度範圍
 *
 * 用於不透明度的圖像取自：
 * 專案/img/system/VisibilityRange1.png
 * 此圖片必須有透明，你可以更改插件內的圖片檔案名稱
 *
 * 詳細說明:
 * 左上角像素必須為黑色
 * 範例圖片 使用黑色的背景以及模糊透明圓圈
 * 不是黑色將視為圖片周邊顯示
 * ----------------------------------------------------------------------------
 * 插件指令
 * ----------------------------------------------------------------------------
 *   VISRANGE imageName           // 改變光圈圖片
 *                                // /img/system/ 在你專案內的資料夾
 *                                // 確保大小寫使用正確
 *
 * 範例:
 * VISRANGE VisibilityRange2
 */

//-----------------------------------------------------------------------------
//  CODE STUFFS
//-----------------------------------------------------------------------------

(function() {

// GALV'S PLUGIN MANAGEMENT. INCLUDED IN ALL GALV PLUGINS THAT HAVE PLUGIN COMMAND CALLS, BUT ONLY RUN ONCE.
if (!Galv.aliased) {
	var Galv_Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
	Game_Interpreter.prototype.pluginCommand = function(command, args) {
		if (Galv.pCmd[command]) {
			Galv.pCmd[command](args);
			return;
		};
		Galv_Game_Interpreter_pluginCommand.call(this, command, args);
	};
	Galv.aliased = true; // Don't keep aliasing for other Galv scripts.
};

// Direct to Plugin Object
Galv.pCmd.VISRANGE = function(arguments) {
	Galv.VR.setImage(arguments);
};
// END GALV'S PLUGIN MANAGEMENT


Galv.VR.setImage = function(image) {
	$gameSystem.galvVis = image;
};


Galv.VR.zoom = Number(PluginManager.parameters('Galv_VisibilityRange')["Zoom Variable"]);
Galv.VR.opacity = Number(PluginManager.parameters('Galv_VisibilityRange')["Opacity Variable"]);
Galv.VR.z = Number(PluginManager.parameters('Galv_VisibilityRange')["Z Position"]);

Game_System.prototype.galvVis = 'VisibilityRange1';

//-----------------------------------------------------------------------------
// Spriteset_Map

var Galv_Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
Spriteset_Map.prototype.createLowerLayer = function() {
	Galv_Spriteset_Map_createLowerLayer.call(this);
	this.setVisibilityRange();
};

Spriteset_Map.prototype.setVisibilityRange = function() {
	this._galvVisRange = new Sprite_GalvVisRange();
	this._tilemap.addChild(this._galvVisRange);
};


//-----------------------------------------------------------------------------
// Sprite_GalvVisRange

function Sprite_GalvVisRange() {
    this.initialize.apply(this, arguments);
}

Sprite_GalvVisRange.prototype = Object.create(Sprite.prototype);
Sprite_GalvVisRange.prototype.constructor = Sprite_GalvVisRange;

Sprite_GalvVisRange.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
	this.name = null;
	this.opacity = 0;
    this.update();
};

Sprite_GalvVisRange.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (this.name != $gameSystem.galvVis) this.loadBitmap();  // If image changed, reload bitmap
	this.opacity = $gameVariables.value(Galv.VR.opacity);
	if (this.opacity <= 0) return;
	
	this.x = $gamePlayer.screenX();
	this.y = $gamePlayer.screenY() - 24;
	var zoom = Math.max($gameVariables.value(Galv.VR.zoom) * 0.01 + 0.5,0.5);
	this.scale.x = zoom;
	this.scale.y = zoom;
};

Sprite_GalvVisRange.prototype.loadBitmap = function() {
	var img = ImageManager.loadSystem($gameSystem.galvVis);
	if (img.isReady()) {
		if (this.bitmap) {
			this._destroyCachedSprite();
			this.bitmap = null;
		};
		
		// Middle Graphic
		var tempSprite = new Sprite();
		tempSprite.bitmap = ImageManager.loadSystem($gameSystem.galvVis);
		var iw = tempSprite.bitmap.width;
		var ih = tempSprite.bitmap.height;
		var color = tempSprite.bitmap.getPixel(0,0);
		
		// Background Color
		this.bitmap = new Bitmap(Graphics.boxWidth * 4,Graphics.boxHeight * 4);
		this.bitmap.fillRect(0, 0, Graphics.boxWidth * 4, Graphics.boxHeight * 4, color);
	
		// Position
		this.anchor.x = 0.5;
		this.anchor.y = 0.5;
		this.x = $gamePlayer.screenX();
		this.y = $gamePlayer.screenY();
		this.z = Galv.VR.z;
	
		// Join Bitmaps
		var cx = ((Graphics.boxWidth * 4) / 2) - (iw / 2);
		var cy = ((Graphics.boxHeight * 4) / 2) - (ih / 2);
		this.bitmap.clearRect(cx, cy, iw, ih);
		this.bitmap.blt(tempSprite.bitmap, 0, 0, iw, ih, cx, cy);

		this.name = $gameSystem.galvVis;
	};
};
})();